<?php

namespace App\Http\Controllers\Api\Admin\V1_0;

use App\Http\Controllers\Controller;
use App\Http\Utils\Miscellaneous;
use App\Http\Utils\MyLog;
use App\Models\Admin\Reseller;
use App\Models\ResellerCompany;
use App\User;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Lang;
use Validator;


class ResellerController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        $query = Reseller::query();

        $inputs = Input::get();
        foreach ($inputs as $key => $value) {
            if ($key != 'page' && $key != 'pagination_per_page') {
                if ($key == 'first_name' || $key == 'last_name')
                    $query->where('resellers.' . $key,'like', '%'.$value.'%');
                else
                    $query->where('resellers.' . $key,$value);
            }
        }

        if (isset($inputs['pagination_per_page']))
            return $query->with('user', 'company', 'company.client')->paginate($inputs['pagination_per_page']);
        else if (!isset($inputs['pagination_per_page']) && isset($inputs['page']))
            return $query->with('user', 'company', 'company.client')->paginate();
        else if (!isset($inputs['pagination_per_page']) && !isset($inputs['page']))
            return $query->with('user', 'company', 'company.client')->get();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        $warning = null;
        $status = null;

        $validator = Validator::make($request->all(), [
            'user.login' => 'required|min:3|max:255',
            'user.password' => 'required|min:6|max:60',
            'company_id' => 'required|integer',
            'first_name' => 'required|min:3|max:20',
            'last_name' => 'required|min:3|max:20',
            'mobile' => 'digits:10'
        ]);

        if ($validator->fails()){
            $status = 400;
            $warning = 'validation_error';
            $messages = $validator->errors()->all();
            MyLog::Warning($status, $warning, null, $messages);
            return response()->json(['error' => $warning, 'messages' => $messages], $status);
        }

        $user_stored = User::where('email', $request->input('user.login'))->first();
        if ($user_stored != null)
            $warning = 'user_already_exist_email';

        if ($warning != null) {
            $status = 409;
            $message = Lang::get('messages.'.$warning);
            MyLog::Warning($status, $warning, $message);
            return response()->json(['error' => $warning, 'message' => $message], $status);
        }

        DB::transaction(function () use ($request) {
            // User information storage
            $user = new User();
            if ($request->has('user')) {
                $user->email = $request->input('user.login');
                $user->password = Hash::make($request->input('user.password'));
                $user->name = '';
                $user->is_reseller = 1;
                $user->verified = 1;
                $user->save();
            }

            // Individual information storage
            $reseller = new Reseller();
            $inputs = $request->all();
            foreach ($inputs as $key => $value) {
                if ($key != 'user')
                    $reseller[$key] = $value;
            }

            $user->reseller()->save($reseller);
            $reseller->save();

            Miscellaneous::incrementVersion();
        });
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
        $reseller = Reseller::findOrFail($id);
        $reseller->user;
        $reseller->company;
        $reseller->company->discounts;

        return $reseller;
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
        $warning = null;
        $status = null;

        $validator = Validator::make($request->all(), [
            'user.login' => 'min:3|max:255',
            'user.password' => 'min:6|max:60',
            'company_id' => 'integer',
            'first_name' => 'min:3|max:20',
            'last_name' => 'min:3|max:20',
            'mobile' => 'digits:10'
        ]);

        if ($validator->fails()){
            $status = 400;
            $warning = 'validation_error';
            $messages = $validator->errors()->all();
            MyLog::Warning($status, $warning, null, $messages);
            return response()->json(['error' => $warning, 'messages' => $messages], $status);
        }

        $reseller = Reseller::findOrFail($id);
        $user = $reseller->user;
        if (Input::has('user.email')) {
            $user->email = $request->input('user.email');
            $users_stored = User::where('email', $user->email);
            $users_stored_count = $users_stored->count();
            if ($users_stored_count == 1) {
                if ($user->id != $users_stored->first()->id)
                    $warning = 'user_already_exist_email';
            } else if ($users_stored_count > 1)
                $warning = 'user_already_exist_email';
            if ($warning != null) {
                $status = 409;
                $message = Lang::get('messages.' . $warning);
                MyLog::Warning($status, $warning, $message);
                return response()->json(['error' => $warning, 'message' => $message], $status);
            }
        }

        DB::transaction(function () use ($request, $reseller) {
            // User information storage
            $user = $reseller->user;
            if ($request->has('user.login'))
                $user->email = $request->input('user.login');
            if ($request->has('user.password'))
                $user->password = Hash::make($request->input('user.password'));
            $user->save();

            // Reseller information storage
            $inputs = $request->all();
            foreach ($inputs as $key => $value) {
                if ($key != 'user')
                    $reseller[$key] = $value;
            }
            $user->reseller()->save($reseller);
            $reseller->save();

            Miscellaneous::incrementVersion();
        });
    }

    /**
     * Delete the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
        $warning = null;
        $status = null;

        $reseller = Reseller::findOrFail($id);
        $user = $reseller->user;

        DB::transaction(function () use ($reseller, $user) {
            // Reseller information storage
            $reseller->delete();
            // User information storage
            $user->delete();
        });
    }

}
