<?php
/**
 * Created by PhpStorm.
 * User: KBM-Adil_Zaid
 * Date: 11/04/2017
 * Time: 14:38
 */

namespace App\Http\Controllers\Api\Users\Resellers\V1_0;

use App\Http\Controllers\Api\Users\V1_0\WelcomePictureController;
use App\Models\Constant;
use App\Models\Users\Resellers\Category;
use App\Models\Users\Resellers\DeliveryAddress;
use App\Models\Users\Resellers\Message;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Log;
use App\Http\Utils\Miscellaneous;

class CatalogController
{
    protected $category;

    public function __construct(Category $category){
        $this->category = $category;
    }

    /**
     * Display the whole catalog at once.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        $categories = Category::where(['deactivated' => 0])->
                            has('groups')->
                            with('groups','groups.products', 'groups.pictures', 'groups.products.unit', 'groups.products.type', 'groups.products.vat', 'groups.products.variants')->
                            get();

        return response()->json(['version' => $this->version(), 'categories' => $categories], 200);
    }

    /**
     * Display all slider, 4 catalogs, messages & delivery adresses.
     *
     * @return \Illuminate\Http\Response
     */
    public function allCatalogs()
    {
        //
        $user = Auth::user();
        $client_id = $user->reseller->company->client->id;
        $this->category->client_id = $client_id;

        $standardProductsCategories = $this->getStandardProductsCategories($client_id);
        $nonStandardProductsCategories = $this->getNonStandardProductsCategories($client_id);
        $specificProductsCategories = $this->getSpecificProductsCategories($client_id);
        $favouriteProductsCategories = $this->getFavouriteProductsCategories($client_id);
        $mesages = $this->getMessages($client_id);
        $deliveryAddresses = $this->getDeliveryAddresses($client_id);
        $welcomePictures = WelcomePictureController::getWelcomePictures();

        return response()->json(['version' => $this->version(),
                                 'data' => ['welcomePictures' => $welcomePictures,
                                            'catalogs' => ['favouriteProductsCategories' => $favouriteProductsCategories,
                                                            'standardProductsCategories' => $standardProductsCategories,
                                                            'nonStandardProductsCategories' => $nonStandardProductsCategories,
                                                            'specificProductsCategories' => $specificProductsCategories]
                                                            ],
                                            'messages' => $mesages,
                                            'deliveryAddresses' => $deliveryAddresses],
                                200);
    }

    /**
     * Display Standard Products Categories with Standard Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function standardProductsCategories()
    {
        //
        $user = Auth::user();
        $client_id = $user->reseller->company->client->id;
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
                                    ->Has('standardProductsGroups')
                                    ->with('standardProductsGroups')->get();
        return response()->json(['version' => $this->version(), 'categories' => $categories], 200);
    }

    /**
     * Display Non Standard Products Categories with Non Standard Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function nonStandardProductsCategories()
    {
        //
        $user = Auth::user();
        $client_id = $user->reseller->company->client->id;
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
                            ->has('nonStandardProductsGroups')
                            ->with('nonStandardProductsGroups')->get();
        return response()->json(['version' => $this->version(), 'categories' => $categories], 200);
    }

    /**
     * Display Specific Products Categories with specific Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function specificProductsCategories()
    {
        //
        $user = Auth::user();
        $client_id = $user->reseller->company->client->id;
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
              ->Has('specificProductsGroups')->with('specificProductsGroups')
              ->get();
        return response()->json(['version' => $this->version(), 'categories' => $categories], 200);
    }


    /**
     * Display Favourite Products Categories with favourite Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function favouriteProductsCategories()
    {
        //
        $user = Auth::user();
        $client_id = $user->reseller->company->client->id;
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
            ->Has('favouriteProductsGroups')->with('favouriteProductsGroups')
            ->get();
        return response()->json(['version' => $this->version(), 'categories' => $categories], 200);
    }

    /**
     * Gt Standard Products Categories with Standard Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function getStandardProductsCategories($client_id)
    {
        //
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
            ->Has('standardProductsGroups')
            ->with('standardProductsGroups')->get();
        return $categories;
    }

    /**
     * Get Non Standard Products Categories with Non Standard Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function getNonStandardProductsCategories($client_id)
    {
        //
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
            ->has('nonStandardProductsGroups')
            ->with('nonStandardProductsGroups')->get();
        return $categories;
    }

    /**
     * Get Specific Products Categories with specific Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function getSpecificProductsCategories($client_id)
    {
        //
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
            ->Has('specificProductsGroups')->with('specificProductsGroups')
            ->get();
        return $categories;
    }


    /**
     * Get Favourite Products Catalog with favourite Products Groups.
     *
     * @return \Illuminate\Http\Response
     */
    public function getFavouriteProductsCategories()
    {
        //
        $user = Auth::user();
        $client_id = $user->reseller->company->client->id;
        $this->category->client_id = $client_id;
        $categories = $this->category->where(['deactivated' => 0])
            ->Has('favouriteProductsGroups')->with('favouriteProductsGroups')
            ->get();
        return $categories;
    }

    /**
     * Get Messages for the client.
     *
     * @return \Illuminate\Http\Response
     */
    public function getMessages($client_id)
    {
        //
        // End date is set
        $messages = Message::where('deactivated', 0)
                                ->where(function ($query) {
                                    $query->where('start_date',  '<=', Carbon::now())
                                        ->orWhereNull('start_date');
                                })
                                ->where(function ($query) {
                                    $query->where('end_date',  '>=', Carbon::now())
                                        ->orWhereNull('end_date');
                                })
                                ->where(function ($query) use ($client_id) {
                                    $query->where('client_id', $client_id)
                                          ->orWhereNull('client_id');
                                })
                                ->get();
        return $messages;
    }

    /**
     * Get Delivery Adresses of the client.
     *
     * @return \Illuminate\Http\Response
     */
    public function getDeliveryAddresses($client_id)
    {
        //
        return DeliveryAddress::where('client_id', $client_id)->where('deactivated', 0)->orderBy('by_default', 'desc')->get();
    }

    /**
     * Display the catalog version.
     *
     * @return \Illuminate\Http\Response
     */
    public function version()
    {
        //
        return Miscellaneous::getVersion();
    }
}
