<?php

namespace App\Http\Controllers\Api\V1_0;

use App\Http\Controllers\Controller;
use App\Mail\ResetPassword;
use App\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Mail;
use Validator;

class AuthenticationController extends Controller
{
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        //
        $validator = Validator::make($request->all(), [
            'old_password' => array('required','min:5','max:20','regex:/^[(\d\p{L} !@#$%^&*()_+\-=\[\]{};\':"\\|,.<>\/?)|]+$/u'),
            'new_password' => array('required','min:5','max:20','regex:/^[(\d\p{L} !@#$%^&*()_+\-=\[\]{};\':"\\|,.<>\/?)|]+$/u'),
            'confirm_password' => 'required|same:new_password',
        ]);

        if ($validator->fails())
            return response()->json(['messages' => $validator->errors()->all()], 404);

        $user = Auth::user();
        $inputs = $request->all();
        if (!Hash::check($request->old_password, $user->password))
            return response()->json(['message' => 'Le mot de passe renseigné ne correspond pas a celui enregistré'], 404);

        $user->password = Hash::make($inputs['new_password']);
        $user->save();
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function resetPassword(Request $request)
    {
        //
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails())
            return response()->json(['messages' => $validator->errors()->all()], 404);

        $user = User::where('email', $request->email)->first();
        if ($user == null)
            return response()->json(['message' => 'Il n\'existe aucun compte correspondant à l\'email renseigné'], 404);

        $newPassword = str_random(8);
        $user->password = Hash::make($newPassword);
        $user->save();

        // Send mail
        try {
            Mail::to($user->email)->send(new ResetPassword($newPassword), 'Flexoli: Nouveau mot de passe');
        }
        catch (\Exception $exception) {
            return response()->json(['error' => 'mail_sending_error','message' => Lang::get('messages.mail_sending_error')], 404);
        }
    }

}
