<?php

namespace App\Http\Controllers\Api\Admin\V1_0;

use App\Http\Controllers\Controller;
use App\Http\Utils\Miscellaneous;
use App\Http\Utils\MyLog;
use App\Models\Admin\Product;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Input;
use Log;


class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        $query = Product::query();

        $inputs = Input::get();
        foreach ($inputs as $key => $value) {
            if ($key != 'page' && $key != 'pagination_per_page') {
                if ($key == 'name')
                    $query->where('products.' . $key,'like', '%'.$value.'%');
                else if ($key == 'group_name')
                    $query->join('groups', 'groups.id', 'products.group_id')
                          ->select('products.group_id', 'groups.name as group_name', 'products.id', 'products.name', 'products.ERP_code')
                          ->where('groups.' . 'name','like', '%'.$value.'%');
                else
                    $query->where('products.' . $key,$value);
            }
        }

        if (isset($inputs['pagination_per_page']))
            return $query->with('group', 'unit', 'discount')->paginate($inputs['pagination_per_page']);
        else if (!isset($inputs['pagination_per_page']) && isset($inputs['page']))
            return $query->with('group', 'unit', 'discount')->paginate();
        else if (!isset($inputs['pagination_per_page']) && !isset($inputs['page']))
            return $query->with('group', 'unit', 'discount')->get();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        $warning = null;
        $product_stored = Product::where('ERP_code', $request->ERP_code)->first();
        if ($product_stored != null)
            $warning = 'product_already_exists_ERP_code';
        if ($warning != null) {
            $status = 409;
            $message = Lang::get('messages.'.$warning);
            MyLog::Warning($status, $warning, $message);
            return response()->json(['error' => $warning, 'message' => $message], $status);
        }

        $inputs = $request->all();
        DB::transaction(function () use ($inputs) {
            $product = new product;
            foreach ($inputs as $key => $value) {
                if ($key != 'variants' &&
                    $key != 'external_references'
                )
                    $product[$key] = $value;
            }
            $product->save();

            if (isset($inputs["variants"])) {
                $variantsInput = $inputs["variants"];
                $product->variants()->attach($variantsInput);
            }

            if (isset($inputs["external_references"])){
                $references = $inputs["external_references"];
                foreach ($references as $reference) {
                    $reference = collect($reference);
                    $product->externalReferences()->attach($reference);
                }
            }
            Miscellaneous::incrementVersion();
        });
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
        return Product::products()->with("group", "unit", "type", "vat", "specific", "discount", "variants", "externalReferences")->findOrFail($id);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
        $product = Product::findOrFail($id);
        $warning = null;

        if (Input::has('name'))
            $product->name = Input::get('name');
        if (Input::has('ERP_code'))
            $product->ERP_code = Input::get('ERP_code');
        $products_stored = Product::where('ERP_code', $product->ERP_code);
        $products_stored_count = $products_stored->count();
        if ($products_stored_count == 1) {
            if ($product->id != $products_stored->first()->id)
                $warning = 'product_already_exists_ERP_code';
        } else if ($products_stored_count > 1)
            $warning = 'product_already_exists_ERP_code';

        if ($warning != null) {
            $status = 409;
            $message = Lang::get('messages.'.$warning);
            MyLog::Warning($status, $warning, $message);
            return response()->json(['error' => $warning, 'message' => $message], $status);
        }

        $inputs = $request->all();
        DB::transaction(function () use ($product, $inputs) {
            foreach ($inputs as $key => $value) {
                if ($key != 'variants' &&
                    $key != 'external_references'
                )
                    $product[$key] = $value;
            }
            $product->save();

            if (isset($inputs["variants"])) {
                $variantsInput = $inputs["variants"];
                $product->variants()->detach();
                $product->variants()->attach($variantsInput);
            }

            if (isset($inputs["external_references"])) {
                $references = $inputs["external_references"];
                $references_id_name_map = array();
                foreach ($references as $referenceInput) {
                    $client_id = $referenceInput['client_id'];
                    $reference_name = $referenceInput['reference_name'];
                    $references_label_name_map['reference_name'] = $reference_name;
                    $references_id_name_map[$client_id] = $references_label_name_map;
                }
                $product->externalReferences()->sync($references_id_name_map);
            }

            Miscellaneous::incrementVersion();
        });
    }

}
